const { exec } = require('child_process');
const { promisify } = require('util');
const { existsSync } = require('fs');
const path = require('path');
const logger = require('./logger');
const { findFFprobe } = require('./ffprobe');

const execAsync = promisify(exec);

function parseFrameRate(rFrameRate) {
  if (!rFrameRate || typeof rFrameRate !== 'string') return null;
  const match = rFrameRate.match(/^(\d+(?:\.\d+)?)\/(\d+(?:\.\d+)?)$/);
  if (!match) return null;
  const num = parseFloat(match[1]);
  const den = parseFloat(match[2]);
  if (den === 0) return null;
  return Math.round((num / den) * 100) / 100;
}

function formatDuration(seconds) {
  if (!Number.isFinite(seconds) || seconds < 0) return '—';
  const h = Math.floor(seconds / 3600);
  const m = Math.floor((seconds % 3600) / 60);
  const s = Math.floor(seconds % 60);
  if (h > 0) return `${h}:${String(m).padStart(2, '0')}:${String(s).padStart(2, '0')}`;
  return `${m}:${String(s).padStart(2, '0')}`;
}

function formatFileSize(bytes) {
  if (!Number.isFinite(bytes) || bytes < 0) return '—';
  if (bytes < 1024) return `${bytes} B`;
  if (bytes < 1024 * 1024) return `${(bytes / 1024).toFixed(2)} KB`;
  return `${(bytes / (1024 * 1024)).toFixed(2)} MB`;
}

/**
 * Get detailed video file metadata using ffprobe
 * @param {string} videoPath - Path to the video file
 * @returns {Promise<object|null>} Metadata object or null on failure
 */
async function getVideoMetadata(videoPath) {
  try {
    if (!videoPath || !existsSync(videoPath)) {
      return null;
    }

    const ffprobePath = findFFprobe();
    const escapedPath = JSON.stringify(videoPath);
    const command = `"${ffprobePath}" -v quiet -print_format json -show_format -show_streams ${escapedPath}`;

    let stdout;
    try {
      const result = await execAsync(command);
      stdout = result.stdout;
    } catch (error) {
      stdout = error.stdout || '';
      if (error.code && error.code !== 0) {
        logger.warn('[VideoMetadata] ffprobe failed:', error.message);
        return null;
      }
    }

    if (!stdout || !stdout.trim()) return null;

    let info;
    try {
      info = JSON.parse(stdout);
    } catch {
      return null;
    }

    const format = info.format || {};
    const streams = Array.isArray(info.streams) ? info.streams : [];
    const videoStream = streams.find((s) => s.codec_type === 'video');
    const audioStream = streams.find((s) => s.codec_type === 'audio');

    const metadata = {
      path: videoPath,
      name: path.basename(videoPath),
      format: {
        name: format.format_name || '—',
        longName: format.format_long_name || '—',
        duration: format.duration ? parseFloat(format.duration) : null,
        durationFormatted: format.duration ? formatDuration(parseFloat(format.duration)) : '—',
        size: format.size ? parseInt(format.size, 10) : null,
        sizeFormatted: format.size ? formatFileSize(parseInt(format.size, 10)) : '—',
        bitrate: format.bit_rate ? parseInt(format.bit_rate, 10) : null,
        bitrateFormatted: format.bit_rate ? `${(parseInt(format.bit_rate, 10) / 1000).toFixed(1)} kbps` : '—',
        nbStreams: format.nb_streams || 0
      },
      video: null,
      audio: null,
      streams: []
    };

    if (videoStream) {
      const fps = parseFrameRate(videoStream.r_frame_rate) || parseFrameRate(videoStream.avg_frame_rate);
      metadata.video = {
        codec: videoStream.codec_name || '—',
        codecLongName: videoStream.codec_long_name || '—',
        width: videoStream.width || null,
        height: videoStream.height || null,
        resolution: videoStream.width && videoStream.height ? `${videoStream.width}×${videoStream.height}` : '—',
        fps: fps != null ? fps : '—',
        pixelFormat: videoStream.pix_fmt || '—',
        bitrate: videoStream.bit_rate ? parseInt(videoStream.bit_rate, 10) : null,
        bitrateFormatted: videoStream.bit_rate ? `${(parseInt(videoStream.bit_rate, 10) / 1000).toFixed(1)} kbps` : '—',
        aspectRatio: videoStream.display_aspect_ratio || videoStream.sample_aspect_ratio || '—'
      };
    }

    if (audioStream) {
      metadata.audio = {
        codec: audioStream.codec_name || '—',
        codecLongName: audioStream.codec_long_name || '—',
        sampleRate: audioStream.sample_rate ? `${parseInt(audioStream.sample_rate, 10)} Hz` : '—',
        channels: audioStream.channels || audioStream.channel_layout || '—',
        bitrate: audioStream.bit_rate ? parseInt(audioStream.bit_rate, 10) : null,
        bitrateFormatted: audioStream.bit_rate ? `${(parseInt(audioStream.bit_rate, 10) / 1000).toFixed(1)} kbps` : '—'
      };
    }

    streams.forEach((s, i) => {
      const type = s.codec_type || 'unknown';
      const codec = s.codec_name || '—';
      const extra = type === 'video' && s.width && s.height ? ` ${s.width}×${s.height}` : '';
      metadata.streams.push({ index: i, type, codec, extra });
    });

    return metadata;
  } catch (error) {
    logger.error('[VideoMetadata] Error:', error.message);
    return null;
  }
}

module.exports = {
  getVideoMetadata
};
